# [`gltest`][]

A [C++11][]/[OpenGL][] \>=[1.0][] [testing][] library.

[`gltest`]: https://git.rcrnstn.net/rcrnstn/gltest
[C++11]: https://en.wikipedia.org/wiki/C++11
[OpenGL]: https://en.wikipedia.org/wiki/OpenGL
[1.0]: https://en.wikipedia.org/wiki/OpenGL#Version_history

## Usage

```cpp
#include <cerrno>
#include <cstring>
#include <fstream>
#include <stdexcept>
#include <string>

#include <GL/glew.h>

#include <glbackend.hpp>
#include <gltest.hpp>


// Function that throws with a system-supplied message.
void file_open(std::string const & filename)
{
    auto file = std::ifstream{"nonexistent"};
    if (!file)
        throw std::runtime_error(std::string{} +
            "Could not open file '" + filename + "':\n" +
            std::strerror(errno) + "." // NOLINT
        );
}


GLTEST(gltest, 640, 480, 2, 0)
{
    backend.prefix("assets/tests");

    GLTEST_EXPECT_VALUE(
        GL_FALSE,
        glIsEnabled(GL_DEPTH_TEST)
    )
    GLTEST_EXPECT_EXCEPTION(false,
        "Expected exception",
        throw std::runtime_error("Expected exception")
    )
    GLTEST_EXPECT_EXCEPTION(true,
        "Could not open file 'nonexistent':\n",
        // "No such file or directory"
        file_open("nonexistent")
    )

    constexpr auto size = 0.5;
    glBegin(GL_TRIANGLE_STRIP);
    glVertex3f(-size, -size, 0);
    glVertex3f(-size, +size, 0);
    glVertex3f(+size, -size, 0);
    glVertex3f(+size, +size, 0);
    glEnd();
    backend.tga_compare("GLTest.tga");
}
```

A `main()` entry point is provided which calls each test declared with the
macro `GLTEST(VERSION_MAJOR, VERSION_MINOR, WIDTH, HEIGHT, NAME)`. A double
buffered OpenGL debug context compatible with the specified version and with a
window of the specified width and height is set up prior to each test run and
teared down afterwards.

If OpenGL \>=4.3 or the extension [`KHR_debug`][] is available,
non-notification debug messages throw errors unless disabled by calling `bool
gltest_debug(false)` (which returns the old value). The root directory for
reading and writing files is set with `void gltest_root(std::string)`. Timing
can be manipulated with `void gltest_set_time(float)` and `float
gltest_get_time()`. Swapping buffers is performed with `void
gltest_swap_buffers()`.

The macro `GLTEST_EXPECT_VALUE(EXPR, VALUE)` runs `EXPR`, compares the result
to `VALUE` and throws an error if they do not match.

The macro `GLTEST_EXPECT_EXCEPTION(PREFIX, EXPR, WHAT)` catches any
[`std::exception`][]-derived exception thrown by `EXPR`, compares its
[`what()`][] to `WHAT` and throws an error if they do not match. If `PREFIX` is
true it is only required that `WHAT` is a prefix of `what()`, to allow trailing
external messages of unknown format. An error is also thrown if `EXPR` throws
no exception at all. `gltest_debug(false)` is automatically called before
`EXPR` is run, the previous value is restored afterwards.

The macro `GLTEST_EXPECT_FRAME(PATH)` reads `PATH` from disk, compares it to
the current (back) framebuffer and throws an error if they do not match. If
`PATH` does not exist, the current framebuffer is instead written to it and a
warning is printed.

If a `GLTEST_EXPECT_*` fails, the values of `GL_VENDOR`, `GL_RENDERER`,
`GL_VERSION` and `GL_SHADING_LANGUAGE_VERSION` is printed to standard error. In
addition, `GLTEST_EXPECT_VALUE` or `GLTEST_EXPECT_EXCEPTION` print something in
the style of:

```
>>> Expression:
glIsEnabled(GL_DEPTH_TEST)
>>> Expected value:
GL_TRUE = 1
>>> Got:
0
```

[`std::exception`]: https://en.cppreference.com/w/cpp/error/exception
[`what()`]: https://en.cppreference.com/w/cpp/error/exception/what
[`KHR_debug`]: https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_debug.txt

## Build system

This project supports [CMake][] and uses [`cmake-common`][]. There are several
ways to use it in other CMake-based projects:

-   With [`find_package`][]: ([Package][] and) [install][] it on the system.

-   With [`add_subdirectory`][]: Bundle it.

-   With [`FetchContent`][]: Download it as part of the CMake configure step.

-   With [`cmake-common`][]: Use any of the above methods through a simplified
    interface.

As usual, use [`add_dependencies`][] or [`target_link_libraries`][] (or
`cmake-common`'s `DEPENDENCIES_*`) to declare the dependency.

[CMake]: https://cmake.org
[`cmake-common`]: https://git.rcrnstn.net/rcrnstn/cmake-common
[`FetchContent`]: https://cmake.org/cmake/help/v3.14/module/FetchContent.html
[`add_subdirectory`]: https://cmake.org/cmake/help/v3.14/command/add_subdirectory.html
[`find_package`]: https://cmake.org/cmake/help/v3.14/command/find_package.html
[Package]: #Package
[Install]: #Install
[`add_dependencies`]: https://cmake.org/cmake/help/v3.14/command/add_dependencies.html
[`target_link_libraries`]: https://cmake.org/cmake/help/v3.14/command/target_link_libraries.html

### Configure and generate

To configure and generate a build tree, use `cmake`:

```sh
cmake -B _build
```

To set the build type, pass e.g. `-D`[`CMAKE_BUILD_TYPE`][]`=Release`.

[`cmake`]: https://cmake.org/cmake/help/v3.14/manual/cmake.1.html#generate-a-project-buildsystem
[`CMAKE_BUILD_TYPE`]: https://cmake.org/cmake/help/v3.14/variable/CMAKE_BUILD_TYPE.html

### Build

To build, use [`cmake --build`][]:

```sh
cmake --build _build
```

To disable building tests, pass `-D`[`BUILD_TESTING`][]`=OFF`.

[`cmake --build`]: https://cmake.org/cmake/help/v3.14/manual/cmake.1.html#build-a-project
[`BUILD_TESTING`]: https://cmake.org/cmake/help/v3.14/module/CTest.html

### Test

To run tests, use [`ctest`][]:

```sh
(cd _build && ctest)
```

To show output from failing tests, pass `--output-on-failure`. To show output
from all tests, pass `--verbose`.

[`ctest`]: https://cmake.org/cmake/help/v3.14/manual/ctest.1.html

### Package

To package, use [`cpack`][]:

```sh
(cd _build && cpack)
```

[`cpack`]: https://cmake.org/cmake/help/v3.14/manual/cpack.1.html

### Install

To install onto the current system, use [`cmake --install`][]:

```sh
cmake --install _build
```

To set the prefix, pass e.g. `-D`[`CMAKE_INSTALL_PREFIX`][]`="$HOME/.local"`.

[`cmake --install`]: https://cmake.org/cmake/help/v3.14/manual/cmake.1.html#install-a-project
[`CMAKE_INSTALL_PREFIX`]: https://cmake.org/cmake/help/v3.14/variable/CMAKE_INSTALL_PREFIX.html

## License

Licensed under the [ISC License][] unless otherwise noted, see the
[`LICENSE`][] file.

[ISC License]: https://choosealicense.com/licenses/isc
[`LICENSE`]: LICENSE
